/**
 * MIT License
 * Copyright (c) 2025 Andri Kruus for Maksu GMBH
 * This file is released under the MIT License.
 */
using System.Text;
using System.Text.Json;
using System.Reflection;
using System.Net;
using System.Net.Http;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text.RegularExpressions;

namespace Com.Maksupay.Sdk
{
		/// <summary>
		/// Maksu>SDK by Maksu GMBH version 1.0, 2025/11
		/// </summary>
    public class MaksuSDK
    {
        public enum IfV5RequestParams
        {
            version,
            mid,
            lang,
            trType,
            orderid,
            orderDesc,
            orderAmount,
            currency,
            payerName,
            payerEmail,
            payerPhone,
            billCountry,
            billState,
            billZip,
            billCity,
            billAddress,
            shipCountry,
            shipState,
            shipZip,
            shipCity,
            shipAddress,
            weight,
            dimensions,
            addFraudScore,
            maxPayRetries,
            reject3dsU,
            payMethod,
            blockScore,
            cssUrl,
            confirmUrl,
            cancelUrl,
            extInstallmentoffset,
            extInstallmentperiod,
            extRecurringfrequency,
            extRecurringenddate,
            extXOrderId,
            extTokenOptions,
            extToken,
            var1,
            var2,
            var3,
            var4,
            var5,
            var6,
            var7,
            var8,
            var9
        }

        public enum IfV5ResponseParams
        {
            version,
            mid,
            orderid,
            status,
            orderAmount,
            currency,
            paymentTotal,
            message,
            riskScore,
            payMethod,
            txId,
            paymentRef,
            shipCountry,
            shipState,
            shipZip,
            shipCity,
            shipAddress,
            shipRecipientName,
            shipRecipientPhone,
            extToken,
            extTokenPanEnd,
            extTokenExp,
            extData,
            var1,
            var2,
            var3,
            var4,
            var5,
            var6,
            var7,
            var8,
            var9
        }

        public enum IfV5NotificationParams
        {
            version,
            mid,
            orderid,
            eventType,
            statusBefore,
            status,
            settlStatusBefore,
            settlStatus,
            orderAmount,
            currency,
            paymentTotal,
            payMethod,
            txId,
            paymentRef,
        }
        
        public enum IfV5SigParams
        {
			signature,
			publicKeyHash
		}
        public static readonly string CERT_HEADER = "-----BEGIN CERTIFICATE-----";
        public static readonly string CERT_FOOTER = "-----END CERTIFICATE-----";
        public static readonly string PUB_HEADER = "-----BEGIN PUBLIC KEY-----";
        public static readonly string PUB_FOOTER = "-----END PUBLIC KEY-----";
        public static readonly string PRI_HEADER = "-----BEGIN PRIVATE KEY-----";
        public static readonly string PRI_FOOTER = "-----END PRIVATE KEY-----";


	/// <summary>
	/// Creates a MaksuPay request string from the specified request parameters.
	/// </summary>
	/// <param name="merchantKey">
	/// Merchant privte key in PKCS8 format (pem)
	/// </param>
	/// <param name="merchantCert">
	/// Merchant certficate with public key correspnining to signing private key X509 pem
	/// </param>
	/// <param name="requestParams">
	/// A dictionary containing the request parameters where the key is the
	/// parameter name and the value is the corresponding parameter value.
	/// </param>
	/// <returns>
	/// A string representing the formatted MaksuPay request that can be
	/// sent to the payment gateway with HTTP post.
	/// </returns>
	/// <exception cref="ArgumentNullException">
	/// Thrown when <paramref name="requestParams"/> is null.
	/// </exception>
	/// <remarks>
	/// This method serializes or concatenates the provided parameters
	/// into the format expected by the MaksuPay backend.
	/// </remarks>
        public string CreateMaksuRequest(string merchantKey, string merchantCert, Dictionary<string, string> requestParams)
        {
            StringBuilder request = new StringBuilder(2048);
            StringBuilder sigString = new StringBuilder(2048);
            foreach (IfV5RequestParams p in Enum.GetValues(typeof(IfV5RequestParams)))
            {
                string pv = requestParams[p.ToString()];
                if (!string.IsNullOrEmpty(pv))
                {
                    if (request.Length > 0)
                    {
                        request.Append("&");
                    }
                    request.Append(p.ToString() + "=" + WebUtility.UrlEncode(pv));
                    sigString.Append(pv).Append(";");
                }
            }
            requestParams["signaturebase"] = sigString.ToString();
            byte[] data = Encoding.UTF8.GetBytes(sigString.ToString());
            String sig = MaksuSDK.CalculateSignatureRS256(data, merchantKey);
            String pkHash = MaksuSDK.PublicKeySHA256Hash(merchantCert);
            request.Append("signature=" + WebUtility.UrlEncode(sig));
            request.Append("publicKeyHash=" + WebUtility.HtmlEncode(pkHash));
            return request.ToString();
        }

	/// <summary>
	/// Creates a MaksuPay request as web form to be bosted to maksu shop handler
	/// </summary>
	/// <param name="merchantKey">
	/// Merchant privte key in PKCS8 format (pem)
	/// </param>
	/// <param name="merchantCert">
	/// Merchant certficate with public key correspnining to signing private key X509 pem
	/// </param>
	/// <param name="maksuURL">
	/// Url to post request
	/// Test: https://pay.test.maksupay.com/vpos/shophandler
	/// </param>
	/// <param name="requestParams">
	/// A dictionary containing the request parameters where the key is the
	/// parameter name and the value is the corresponding parameter value.
	/// </param>
	/// <returns>
	/// A string representing the formatted web <frorm... > </form> MaksuPay request that can be
	/// sent to the payment gateway with HTTP browser post.
	/// </returns>
	/// <exception cref="ArgumentNullException">
	/// Thrown when <paramref name="requestParams"/> is null.
	/// </exception>
	/// <remarks>
	/// This method serializes or concatenates the provided parameters
	/// into the format expected by the MaksuPay backend.
	/// </remarks>
        public string CreateMaksuRequestAsForm(string merchantKey, string merchantCert, string maksuURL, Dictionary<string, string> requestParams)
        {
            StringBuilder request = new StringBuilder(3072);
            StringBuilder sigString = new StringBuilder(2048);

            request.Append("<form id=\"maksuPay\" action=\"" + maksuURL + "\" method=\"post\">\n");
            foreach (IfV5RequestParams p in Enum.GetValues(typeof(IfV5RequestParams)))
            {
                string pv = requestParams[p.ToString()];
                if (!string.IsNullOrEmpty(pv))
                {
                    request.Append("<input type=\"hidden\" name=\"").Append(p.ToString()).
                        Append("\" value=\"").Append(WebUtility.HtmlEncode(pv)).Append("\"/>\n");
                    sigString.Append(pv).Append(";");
                }
            }

            requestParams["signaturebase"] = sigString.ToString();
            byte[] data = Encoding.UTF8.GetBytes(sigString.ToString());
            String sig = MaksuSDK.CalculateSignatureRS256(data, merchantKey);
            String pkHash = MaksuSDK.PublicKeySHA256Hash(merchantCert);
            request.Append("<input type=\"hidden\" name=\"signature\" value=\"" + WebUtility.HtmlEncode(sig) + "\"/>\n");
            request.Append("<input type=\"hidden\" name=\"publicKeyHash\" value=\"" + WebUtility.HtmlEncode(pkHash) + "\"/>\n");
            request.Append("</form>\n");
            return request.ToString();
        }

		/// <summary>
		/// Validates a MaksuPay response signature using the provided certificates and response parameters.
		/// </summary>
		/// <param name="maksuCerts">
		/// The public certificate or certificate bundle used to verify the response
		/// signature. Must contain the public key corresponding to the signing key.
		/// </param>
		/// <param name="responseParams">
		/// A dictionary containing the response parameters returned by the MaksuPay
		/// gateway. The dictionary must include the <c>signature</c> (and publicKeyHash recommended if sent) and  field along with
		/// all parameters required to construct the signature base string.
		/// </param>
		/// <returns>
		/// <c>true</c> if the response signature is valid; otherwise, <c>false</c>.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when <paramref name="maksuCerts"/> or <paramref name="responseParams"/> is null.
		/// </exception>
		/// <exception cref="FormatException">
		/// Thrown when the signature value in <paramref name="responseParams"/> is malformed
		/// or cannot be decoded.
		/// </exception>
		/// <remarks>
		/// This method reconstructs the canonical signature base string from the provided
		/// parameters, extracts the signature from <paramref name="responseParams"/>, and
		/// verifies it using the public key extracted from <paramref name="maksuCerts"/>.
		/// </remarks>
        public bool ValidateResponse(string maksuCerts, Dictionary<string, string> responseParams)
        {
			string pattern = @"(-----END CERTIFICATE-----)";
			string[] maksuCertsX=Regex.Split(maksuCerts, pattern);
			string pkHash=responseParams[IfV5SigParams.publicKeyHash.ToString()];
			X509Certificate2[] maksuCertsX509=new X509Certificate2[maksuCertsX.Length];
			for(var i=0; maksuCertsX!=null && i<maksuCertsX.Length; i++)
			{
	            X509Certificate2 x509 = RsaKeyLoader.LoadX509Certificate(maksuCertsX[i]);
	            maksuCertsX509[i]=x509;
	            
            }
            return ValidateResponse(maksuCertsX509,  responseParams);
        }
        
		/// <summary>
		/// Validates a MaksuPay response signature using the provided certificates and response parameters.
		/// </summary>
		/// <param name="maksuCerts">
		/// The array of public certificate or certificate bundle used to verify the response
		/// signature. Must contain the public key corresponding to the signing key.
		/// </param>
		/// <param name="responseParams">
		/// A dictionary containing the response parameters returned by the MaksuPay
		/// gateway. The dictionary must include the <c>signature</c> (and publicKeyHash recommended if sent) and  field along with
		/// all parameters required to construct the signature base string.
		/// </param>
		/// <returns>
		/// <c>true</c> if the response signature is valid; otherwise, <c>false</c>.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when <paramref name="maksuCerts"/> or <paramref name="responseParams"/> is null.
		/// </exception>
		/// <exception cref="FormatException">
		/// Thrown when the signature value in <paramref name="responseParams"/> is malformed
		/// or cannot be decoded.
		/// </exception>
		/// <remarks>
		/// This method reconstructs the canonical signature base string from the provided
		/// parameters, extracts the signature from <paramref name="responseParams"/>, and
		/// verifies it using the public key extracted from <paramref name="maksuCerts"/>.
		/// </remarks>        
        public bool ValidateResponse(X509Certificate2[] maksuCerts, Dictionary<string, string> responseParams)
        {
            StringBuilder sigString = new StringBuilder(2048);
            foreach (IfV5ResponseParams p in Enum.GetValues(typeof(IfV5ResponseParams)))
            {
                if (responseParams.ContainsKey(p.ToString()))
                {
                    string pv = responseParams[p.ToString()];
                    if (!string.IsNullOrEmpty(pv))
                    {
                        sigString.Append(pv).Append(";");
                    }
                }
            }
            responseParams["signaturebase"] = sigString.ToString();
            string pkHash=responseParams[IfV5SigParams.publicKeyHash.ToString()];
          	byte[] data = Encoding.UTF8.GetBytes(sigString.ToString());
          	byte[] sigBytes = Convert.FromBase64String(responseParams["signature"]);

			for(int i=0; i<maksuCerts.Length; i++)
			{
				X509Certificate2 certX=maksuCerts[i];
				string pkHashX=PublicKeySHA256Hash(certX);
				RSA rsaPublicX=certX.GetRSAPublicKey();
				if (pkHash!=null && pkHash==pkHashX)
				{
					responseParams["validatedwith"] = certX.Subject.ToString();
            		return ValidateSignatureRS256(rsaPublicX, data, sigBytes);
				}
				else 
				{
					bool isValidWithX=ValidateSignatureRS256(rsaPublicX, data, sigBytes);
					if (isValidWithX)
					{
						return true;
					}
				}
            }
            
            return false;
        }



        void AppendIfFirst(StringBuilder sb, string a, char? b)
        {
            if (!string.IsNullOrEmpty(a))
            {
                sb.Append(a);
                if (b != null)
                {
                    sb.Append(b.Value);
                }
            }
        }

        public static string GetParam(HttpContext http, string pn)
        {
            string method = http.Request.Method;

            string val = method == "POST"
                ? http.Request.Form[pn].ToString()
                : http.Request.Query[pn].ToString();

            return val ?? "";
        }

        public static string GetParamOrNull(HttpContext http, string pn)
        {
            var req = http.Request;

            if (req.Method == "POST" && req.HasFormContentType)
            {
                if (req.Form.TryGetValue(pn, out var v))
                    return v.ToString();
            }

            if (req.Query.TryGetValue(pn, out var q))
                return q.ToString();


            return null;
        }

		/// <summary>
		/// Calculates a digital signature for the specified data using the provided private key.
		/// </summary>
		/// <param name="data">
		/// The raw byte array containing the data that will be signed.
		/// </param>
		/// <param name="privateKey">
		/// The private key in PEM or compatible string format used to generate the
		/// digital signature.
		/// </param>
		/// <returns>
		/// A Base64-encoded string representing the generated digital signature.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when <paramref name="data"/> or <paramref name="privateKey"/> is null.
		/// </exception>
		/// <exception cref="CryptographicException">
		/// Thrown when the private key is invalid or the signing operation fails.
		/// </exception>
		/// <remarks>
		/// The method computes the signature using RSA and SHA-256 (RS256),
		/// Ensure that the private key corresponds to the public key used to validate
		/// the signature on the receiving side.
		/// </remarks>
        public static string CalculateSignatureRS256(byte[] data, string privateKey)
        {
            try
            {
                using RSA rsa = RsaKeyLoader.LoadPrivateKeyFromPkcs8(privateKey);
                byte[] signatureBytes = rsa.SignData(
                    data,
                    HashAlgorithmName.SHA256,
                    RSASignaturePadding.Pkcs1
                );
                using var signer = RSA.Create();
                signer.ImportParameters(rsa.ExportParameters(true));

                byte[] signature = signer.SignData(data, HashAlgorithmName.SHA256, RSASignaturePadding.Pkcs1);
                string sigStr = Convert.ToBase64String(signature);
                return sigStr;
            }
            catch (Exception ex)
            {
                throw new Exception("Signature calculation failed", ex);
            }
        }


		/// <summary>
		/// Sends MaksuPay API JSON request to the specified HTTPS endpoint with siganture and other needed headers.
		/// </summary>
		/// <param name="apiURL">
		/// The full API endpoint URL (HTTPS) to which the request will be sent.
		/// </param>
		/// <param name="timeout">
		/// The request timeout in milliseconds.  
		/// A <see cref="HttpRequestException"/> or <see cref="TaskCanceledException"/>
		/// may be thrown if the timeout is exceeded.
		/// </param>
		/// <param name="merId">
		/// The merchant identifier assigned by MaksuPay.
		/// </param>
		/// <param name="merchantKey">
		/// The merchant's RSA private key used to generate the request signature.
		/// </param>
		/// <param name="merchantCert">
		/// The merchant’s X.509 certificate corresponding to private key used signing
		/// and optionally include certificate metadata in the request.
		/// </param>
		/// <param name="request">
		/// A dictionary containing the request message to be serialized to JSON
		/// and signed before sending to the MaksuPay API.
		/// Keys represent parameter names, and values represent the parameter values.
		/// </param>
		/// <returns>
		/// The <see cref="HttpResponseMessage"/> returned by the MaksuPay server.
		/// The caller is responsible for reading and disposing the response.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown if <paramref name="apiURL"/> or <paramref name="request"/> is null.
		/// </exception>
		/// <exception cref="HttpRequestException">
		/// Thrown when the HTTP request fails, e.g. connection errors, TLS errors,
		/// or non-success status codes when calling <c>EnsureSuccessStatusCode()</c>.
		/// </exception>
		/// <exception cref="CryptographicException">
		/// Thrown when signing the request using <paramref name="merchantKey"/> fails.
		/// </exception>
		/// <remarks>
		/// This method serializes the request parameters, signs the payload using
		/// the provided RSA private key, attaches certificate information, applies
		/// the timeout settings, and submits the HTTPS POST request to the MaksuPay API.
		/// </remarks>
        public static HttpResponseMessage sendAPIRequest(string apiURL, int timeout, string merId,
        		RSA merchantKey, X509Certificate2 merchantCert, Dictionary<string,object> request, StringBuilder requestHeadersDbg)
        {
				byte[] bytes = JsonSerializer.SerializeToUtf8Bytes(request);
                string sigHeader=CalculateSignatureForJSONHeader(bytes, merchantKey);
                string pkHashHeader=PublicKeySHA256Hash(merchantCert);
				
				var http = new HttpClient();
				http.Timeout = TimeSpan.FromSeconds(timeout);

				ByteArrayContent content = new ByteArrayContent(bytes);
				content.Headers.ContentType = new System.Net.Http.Headers.MediaTypeHeaderValue("application/json");
				content.Headers.ContentLength =bytes.Length;
				content.Headers.Add("X-Payload-Signature", sigHeader);
				content.Headers.Add("X-Public-Key-Hash", pkHashHeader);
				content.Headers.Add("X-Sender-ID", merId);
				if (requestHeadersDbg!=null)
				{
					
					foreach (var header in content.Headers)
					{
						
						if (requestHeadersDbg.Length>0)
						{
							requestHeadersDbg.Append("\n");
					}
					requestHeadersDbg.Append($"{header.Key} = {string.Join(", ", header.Value)}");
					}
				}
				
				HttpResponseMessage response = http.PostAsync(apiURL, content).GetAwaiter().GetResult();
				return response;
		}

        public static HttpResponseMessage sendAPIRequest(string apiURL, int timeout, string merId,
        		string merchantKey, string merchantCert, string request)
        {
			return sendAPIRequest(apiURL, timeout, merId, merchantKey, merchantCert, request, null);
		}

        public static HttpResponseMessage sendAPIRequest(string apiURL, int timeout, string merId,
        		string merchantKey, string merchantCert, string request, StringBuilder requestHeadersDbg)
        {
			RSA merchantKeyObj=RsaKeyLoader.LoadPrivateKeyFromPkcs8(merchantKey); 
			X509Certificate2 merchantCertObj = RsaKeyLoader.LoadX509Certificate(merchantCert);
			Dictionary<string,object> requestObj=JsonSerializer.Deserialize<Dictionary<string,object>>(request);
			return sendAPIRequest(apiURL, timeout, merId, merchantKeyObj, merchantCertObj, requestObj, requestHeadersDbg);
		}

		/// <summary>
		/// Calculates a digital signature for the specified data using the provided private key.
		/// Useful to create payload signature for JSON API message for http header X-Payload-Signature
		/// </summary>
		/// <param name="data">
		/// The raw byte array containing the data (JSON message body) that will be signed.
		/// </param>
		/// <param name="privateKey">
		/// The private key in PEM or compatible string format used to generate the
		/// digital signature.
		/// </param>
		/// <returns>
		/// A Base64-encoded string representing the generated digital signature, prepended with algorithm "RSA-SHA256;"
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when <paramref name="data"/> or <paramref name="privateKey"/> is null.
		/// </exception>
		/// <exception cref="CryptographicException">
		/// Thrown when the private key is invalid or the signing operation fails.
		/// </exception>
		/// <remarks>
		/// The method computes the signature using RSA and SHA-256 (RS256),
		/// Ensure that the private key corresponds to the public key used to validate
		/// the signature on the receiving side.
		/// </remarks>
        public static string CalculateSignatureForJSONHeader(byte[] data, string privateKey)
        {
	        using RSA rsa = RsaKeyLoader.LoadPrivateKeyFromPkcs8(privateKey);	
	        return CalculateSignatureForJSONHeader(data, rsa);
		}
        public static string CalculateSignatureForJSONHeader(byte[] data, RSA privateKey)
        {
            try
            {
                byte[] signatureBytes = privateKey.SignData(
                    data,
                    HashAlgorithmName.SHA256,
                    RSASignaturePadding.Pkcs1
                );
                using var signer = RSA.Create();
                signer.ImportParameters(privateKey.ExportParameters(true));

                byte[] signature = signer.SignData(data, HashAlgorithmName.SHA256, RSASignaturePadding.Pkcs1);
                string sigStr = Convert.ToBase64String(signature);
                return "RSA-SHA256;"+sigStr;
            }
            catch (Exception ex)
            {
                throw new Exception("Signature calculation failed", ex);
            }
        }


        public static bool ValidateApiResponse(string maksuCerts, string pkHash, string sigB64, byte[] data)
        {
			string pattern = @"(-----END CERTIFICATE-----)";
			string[] maksuCertsX=Regex.Split(maksuCerts, pattern);
			X509Certificate2[] maksuCertsX509=new X509Certificate2[maksuCertsX.Length];
			for(var i=0; maksuCertsX!=null && i<maksuCertsX.Length; i++)
			{
	            X509Certificate2 x509 = RsaKeyLoader.LoadX509Certificate(maksuCertsX[i]);
	            maksuCertsX509[i]=x509;
	            
            }
            return ValidateApiResponse(maksuCertsX509, pkHash, sigB64, data);
        }
        
        public static bool ValidateApiResponse(X509Certificate2[] maksuCerts, string pkHash, string sigB64, byte[] data)
        {
			for(int i=0; i<maksuCerts.Length; i++)
			{
				X509Certificate2 certX=maksuCerts[i];
				string pkHashX=PublicKeySHA256Hash(certX);
				RSA rsaPublicX=certX.GetRSAPublicKey();
				if (pkHash!=null && pkHash==pkHashX)
				{
					return ValidateSignatureJSONHeader(rsaPublicX, sigB64, data);
				}
				else 
				{
					bool isValidWithX=ValidateSignatureJSONHeader(rsaPublicX, sigB64, data);
					if (isValidWithX)
					{
						return true;
					}
				}
            }
            return false;
        }

		
		public static string GetVersion()
		{
			return Assembly.GetExecutingAssembly().GetName().Version!.ToString();
		}


		/// <summary>
		/// Validates a digital signature against the specified data using the provided RSA public key.
		/// </summary>
		/// <param name="publicKey">
		/// The RSA public key used to verify the signature.  
		/// Must correspond to the private key that was used to generate the signature.
		/// </param>
		/// <param name="data">
		/// The raw byte array representing the original data that was signed.
		/// </param>
		/// <param name="signature">
		/// The digital signature to validate, provided as a raw byte array
		/// </param>
		/// <returns>
		/// <c>true</c> if the signature is valid for the given data and public key;
		/// otherwise, <c>false</c>.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when any of the parameters (<paramref name="publicKey"/>,
		/// <paramref name="data"/>, or <paramref name="signature"/>) is null.
		/// </exception>
		/// <exception cref="CryptographicException">
		/// Thrown when the verification process fails due to an invalid key, incorrect format,
		/// unsupported algorithm, or other cryptographic-related issues.
		/// </exception>
		/// <remarks>
		/// This method performs RSA verification using SHA-256 (RS256),
		/// depending on the configuration of the provided <see cref="RSA"/> instance.
		/// Ensure that the <paramref name="signature"/> bytes match the signing algorithm and
		/// </remarks>
        public static bool ValidateSignatureRS256(RSA publicKey, byte[] data, byte[] signature)
        {
            return publicKey.VerifyData(
                    data,
                    signature,
                    HashAlgorithmName.SHA256,
                    RSASignaturePadding.Pkcs1);
        }
        
	/// <summary>
		/// Validates a digital signature against the specified data using the provided RSA public key.
		/// </summary>
		/// <param name="publicKey">
		/// The RSA public key used to verify the signature.  
		/// Must correspond to the private key that was used to generate the signature.
		/// </param>
		/// <param name="data">
		/// The raw byte array representing the original data that was signed.
		/// </param>
		/// <param name="xPayloadSignture">
		/// The digital signature as in X-Payload-Signature header to validate, provided assting Base64-encoded.
		/// </param>
		/// <returns>
		/// <c>true</c> if the signature is valid for the given data and public key;
		/// otherwise, <c>false</c>.
		/// </returns>
		/// <exception cref="ArgumentNullException">
		/// Thrown when any of the parameters (<paramref name="publicKey"/>,
		/// <paramref name="data"/>, or <paramref name="signature"/>) is null.
		/// </exception>
		/// <exception cref="CryptographicException">
		/// Thrown when the verification process fails due to an invalid key, incorrect format,
		/// unsupported algorithm, or other cryptographic-related issues.
		/// </exception>
		/// <remarks>
		/// This method performs RSA verification using SHA-256 (RS256),
		/// depending on the configuration of the provided <see cref="RSA"/> instance.
		/// Ensure that the <paramref name="signature"/> bytes match the signing algorithm and
		/// padding mode used to generate the signature.
		/// </remarks>        
        public static bool ValidateSignatureJSONHeader(RSA publicKey, string xPayloadSignture, byte[] data)
        {
			HashAlgorithmName algName=HashAlgorithmName.SHA256;

			if (xPayloadSignture.StartsWith("RSA-SHA256;"))
			{
				algName=HashAlgorithmName.SHA256;
			}
			else if (xPayloadSignture.StartsWith("RSA-SHA512;"))
			{
				algName=HashAlgorithmName.SHA512;
			}
			byte[] signature=Convert.FromBase64String(xPayloadSignture.Substring(11));
			
            return publicKey.VerifyData(
                    data,
                    signature,
                    algName,
                    RSASignaturePadding.Pkcs1);
        }

        public static string PublicKeySHA256Hash(string certficate)
        {
            X509Certificate2 certX = RsaKeyLoader.LoadX509Certificate(certficate);
            return PublicKeySHA256Hash(certX);
        }

        public static string PublicKeySHA256Hash(X509Certificate2 cert)
        {
			RSA rsaPublic=cert.GetRSAPublicKey();
            byte[] publicKeyBytes = rsaPublic.ExportSubjectPublicKeyInfo();
            using (var sha = SHA256.Create())
            {
                byte[] digest = sha.ComputeHash(publicKeyBytes);
                return Convert.ToBase64String(digest);
            }
        }

        // end class
    }

    public class RsaKeyLoader
    {
        /// <summary>
        /// Loads an RSA private key from a PKCS#8 PEM string.
        /// </summary>
        public static RSA LoadPrivateKeyFromPkcs8(string pkcs8Pem)
        {
            var pem = pkcs8Pem
                .Replace(MaksuSDK.PRI_HEADER, "")
                .Replace(MaksuSDK.PRI_FOOTER, "")
                .Replace("\r", "")
                .Replace("\n", "")
                .Trim();

            byte[] privateKeyBytes = Convert.FromBase64String(pem);
            RSA rsa = RSA.Create();
            rsa.ImportPkcs8PrivateKey(privateKeyBytes, out _);
            return rsa;
        }

        /// <summary>
        /// Loads an RSA public key from a PEM string
        /// </summary>
        public static RSA LoadPublicKeyFromPEM(string pubPem)
        {
            var pem = pubPem
                .Replace(MaksuSDK.PUB_HEADER, "")
                .Replace(MaksuSDK.PUB_FOOTER, "")
                .Replace("\r", "")
                .Replace("\n", "")
                .Trim();

            byte[] publicKeyBytes = Convert.FromBase64String(pubPem);
            RSA rsa = RSA.Create();
            rsa.ImportSubjectPublicKeyInfo(publicKeyBytes, out _);
            return rsa;
        }

        public static RSA LoadPublicKeyFromX509Certificate(string certPem)
        {
            var cert = LoadX509Certificate(certPem);
            return cert.GetRSAPublicKey();
        }
        
        public static X509Certificate2 LoadX509Certificate(string certPem)
        {
            string base64 = certPem
                .Replace(MaksuSDK.CERT_HEADER, "")
                .Replace(MaksuSDK.CERT_FOOTER, "")
                .Replace("\r", "")
                .Replace("\n", "")
                .Trim();

            byte[] certBytes = Convert.FromBase64String(base64);
            return new X509Certificate2(certBytes);

        }
    } // class

} // namespace